<?php
// app/Models/BaseRepository.php
namespace App\Models;

use App\Core\Database;

/**
 * Základní repository poskytující mapování řádku na entity.
 * Postup: potomci předají do parent konstruktoru název tabulky a třídu entity.
 */
abstract class BaseRepository
{
    protected \PDO $pdo;
    protected string $table;
    protected string $entityClass;

    public function __construct(Database $db, string $table, string $entityClass)
    {
        $this->pdo = $db->pdo();
        $this->table = $table;
        $this->entityClass = $entityClass;
    }

    /**
     * Mapuje asociativní pole (řádek z DB) na instanci entity.
     * Používá metodu fill() v App\Core\Model.
     *
     * @param array $row
     * @return object instance třídy $this->entityClass
     */
    protected function mapRowToEntity(array $row)
    {
        $class = $this->entityClass;
        $entity = new $class();
        if (method_exists($entity, 'fill')) {
            $entity->fill($row);
        } else {
            // fallback: při chybě nastavíme vlastnosti dynamicky
            foreach ($row as $k => $v) {
                $entity->$k = $v;
            }
        }
        return $entity;
    }

    /**
     * Vrátí pole entit.
     *
     * @return array<object>
     */
    public function findAll(): array
    {
        $stmt = $this->pdo->query("SELECT * FROM `" . $this->table . "`");
        $rows = $stmt->fetchAll();
        return array_map(fn($r) => $this->mapRowToEntity($r), $rows);
    }

    /**
     * Najde entitu podle ID, nebo vrátí null.
     *
     * @param int $id
     * @return object|null
     */
    public function find(int $id)
    {
        $stmt = $this->pdo->prepare("SELECT * FROM `" . $this->table . "` WHERE id = :id LIMIT 1");
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch();
        return $row ? $this->mapRowToEntity($row) : null;
    }

    /**
     * Smaže záznam podle ID.
     */
    public function delete(int $id): bool
    {
        // TODO: odkomentovat před nasazením na produkci
       // $stmt = $this->pdo->prepare("DELETE FROM `" . $this->table . "` WHERE id = :id");
       // return $stmt->execute(['id' => $id]);
       return false;
    }

    public function findByCategory(string $category): array
    {
        $stmt = $this->pdo->prepare("SELECT * FROM `" . $this->table . "` WHERE category = :category");
        $stmt->execute(['category' => $category]);
        $rows = $stmt->fetchAll();
        return array_map(fn($r) => $this->mapRowToEntity($r), $rows);
    }
}