<?php 
require 'database.php';

// Získání minimální a maximální ceny z databáze
$priceQuery = $pdo->query("SELECT MIN(price) AS min_price, MAX(price) AS max_price FROM bikes");
$priceRange = $priceQuery->fetch(PDO::FETCH_ASSOC);
$minPrice = (int)$priceRange['min_price'];
$maxPrice = (int)$priceRange['max_price'];

// Načtení značek a kategorií
$brandStmt = $pdo->query("SELECT * FROM brands");
$brands = $brandStmt->fetchAll(PDO::FETCH_ASSOC);

$categoryStmt = $pdo->query("SELECT * FROM categories");
$categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);

// Vybrané filtry
$selectedCategory = isset($_GET['category']) ? (int) $_GET['category'] : 0;
$selectedBrand = isset($_GET['brand']) ? (int) $_GET['brand'] : 0;
$filterMin = isset($_GET['min_price']) ? (int) $_GET['min_price'] : $minPrice;
$filterMax = isset($_GET['max_price']) ? (int) $_GET['max_price'] : $maxPrice;
$sortOption = isset($_GET['sort']) ? $_GET['sort'] : 'price_asc';

$params = [
    'min_price' => $filterMin,
    'max_price' => $filterMax
];

$sql = "SELECT bikes.id, bikes.model, bikes.color, bikes.price, bikes.image_main, 
        brands.Brand AS brand_name, categories.Category AS category_name
        FROM bikes
        JOIN brands ON bikes.brand_id = brands.id
        JOIN categories ON bikes.category_id = categories.id
        WHERE bikes.price BETWEEN :min_price AND :max_price";

if ($selectedCategory > 0) {
    $sql .= " AND bikes.category_id = :category_id";
    $params['category_id'] = $selectedCategory;
}

if ($selectedBrand > 0) {
    $sql .= " AND bikes.brand_id = :brand_id";
    $params['brand_id'] = $selectedBrand;
}

switch ($sortOption) {
    case 'price_asc':
        $sql .= " ORDER BY bikes.price ASC";
        break;
    case 'price_desc':
        $sql .= " ORDER BY bikes.price DESC";
        break;
    case 'az':
        $sql .= " ORDER BY bikes.model ASC";
        break;
    case 'za':
        $sql .= " ORDER BY bikes.model DESC";
        break;
    default:
        $sql .= " ORDER BY bikes.price ASC";
}

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$bikes = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>  



<!DOCTYPE html>
<html lang="cs">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Katalog Horských Kol</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/nouislider@15.7.0/dist/nouislider.min.css">
    <script src="https://cdn.jsdelivr.net/npm/nouislider@15.7.0/dist/nouislider.min.js"></script>
</head>

<body>
    <div class="container-fluid">
        <h1 class="my-4">Katalog horských kol</h1>
        <button class="btn btn-outline-dark w-100 mb-3 d-md-none" type="button" data-bs-toggle="collapse" data-bs-target="#filterSidebar">
            Zobrazit / skrýt filtry
        </button>

        <div class="row">
            <!-- Sidebar filtr -->
            <div class="col-md-3">
                <div class="collapse d-md-block show" id="filterSidebar">
                    <div class="filter-box">
                        <form method="GET" class="mb-3">
                            <div class="mb-3">
                                <label for="category" class="form-label">Kategorie:</label>
                                <select name="category" id="category" class="form-select">
                                    <option value="0">Vše</option>
                                    <?php foreach ($categories as $category): ?>
                                        <option value="<?= $category['id'] ?>" <?= $selectedCategory == $category['id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($category['Category']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label for="brand" class="form-label">Značka:</label>
                                <select name="brand" id="brand" class="form-select">
                                    <option value="0">Vše</option>
                                    <?php foreach ($brands as $brand): ?>
                                        <option value="<?= $brand['id'] ?>" <?= $selectedBrand == $brand['id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($brand['Brand']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Cena: <span id="priceOutput"></span></label>
                                    <div id="priceSlider" class="mt-3 mb-3"></div>
                                        <input type="hidden" id="minPriceInput" name="min_price" value="<?= $filterMin ?>">
                                    <input type="hidden" id="maxPriceInput" name="max_price" value="<?= $filterMax ?>">
                            </div>

                            <div class="mb-3">
                                <label for="sort" class="form-label">Třídit podle:</label>
                                    <select name="sort" id="sort" class="form-select">
                                        <option value="price_asc" <?= $sortOption == 'price_asc' ? 'selected' : '' ?>>Cena: od nejlevnější</option>
                                        <option value="price_desc" <?= $sortOption == 'price_desc' ? 'selected' : '' ?>>Cena: od nejdražší</option>
                                        <option value="az" <?= $sortOption == 'az' ? 'selected' : '' ?>>Model: A–Z</option>
                                        <option value="za" <?= $sortOption == 'za' ? 'selected' : '' ?>>Model: Z–A</option>
                                    </select>
                            </div>

                            <button type="submit" class="btn btn-primary w-100 mt-2">Filtrovat</button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Výpis kol -->
            <div class="col-md-9">
                <div class="row">
                    <?php foreach ($bikes as $bike): ?>
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card h-100">
                                <img src="img/<?= htmlspecialchars($bike['image_main']) ?>" class="card-img-top" alt="Obrázek kola">
                                <div class="card-body">
                                    <h5 class="card-title"><?= htmlspecialchars($bike['brand_name']) ?> <?= htmlspecialchars($bike['model']) ?></h5>
                                    <p class="card-text"><strong>Cena:</strong> <?= number_format($bike['price'], 0, ',', ' ') ?> Kč</p>
                                    <a href="detail.php?id=<?= $bike['id'] ?>" class="btn btn-primary">Zobrazit podrobnosti</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <nav class="navbar navbar-light bg-light justify-content-end px-3 mt-4">
        <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#loginModal">
            Admin / Login
        </button>
    </nav>

    <!--LOGIN MODAL -->
    <div class="modal fade" id="loginModal" tabindex="-1" aria-labelledby="loginModallabal1" aria-hidden="true">
        <div class="modal-dialog modal-sm modal-dialog-centered">
            <div class="modal-content">
                <form method="post" action="login.php">
                    <div class="modal-header">
                        <h5 class="modal-title" id="loginModalLabel">Přihlášení</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Zavřít"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="username" class="form-label">Uživatelské jméno</label>
                            <input type="text" class="form-control" id="username" name="username" required>
                        </div>
                        <div class="mb-3">
                            <label for="password" class="form-label">Heslo</label>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                        <button type="submit" class="btn btn-primary w-100">Přihlásit se</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- JavaScript pro slider a hover efekt -->


    <script>
document.addEventListener('DOMContentLoaded', function () {
    // noUiSlider
    var priceSlider = document.getElementById('priceSlider');
    var minPrice = <?= $minPrice ?>;
    var maxPrice = <?= $maxPrice ?>;
    var minStart = <?= $filterMin ?>;
    var maxStart = <?= $filterMax ?>;
    var minInput = document.getElementById('minPriceInput');
    var maxInput = document.getElementById('maxPriceInput');
    var priceOutput = document.getElementById('priceOutput');

    noUiSlider.create(priceSlider, {
        start: [minStart, maxStart],
        connect: true,
        range: {
            'min': minPrice,
            'max': maxPrice
        },
        step: 1000,
        format: {
            to: value => Math.round(value),
            from: value => Number(value)
        }
    });

    priceSlider.noUiSlider.on('update', function (values) {
        minInput.value = values[0];
        maxInput.value = values[1];
        priceOutput.textContent = `${Number(values[0]).toLocaleString('cs-CZ')} Kč – ${Number(values[1]).toLocaleString('cs-CZ')} Kč`;
    });

    // Hover efekt na obrázek kola (zvětšení)
    document.querySelectorAll('.card-img-top').forEach(function(img) {
        img.addEventListener('mouseenter', function() {
            img.style.transform = 'scale(1.08)';
            img.style.transition = 'transform 0.3s cubic-bezier(.4,2,.3,1)';
            img.style.zIndex = 10;
        });
        img.addEventListener('mouseleave', function() {
            img.style.transform = 'scale(1)';
            img.style.transition = 'transform 0.3s cubic-bezier(.4,2,.3,1)';
            img.style.zIndex = 1;
        });
    });
});
</script>
</body>
</html>
